//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 5/10/20.
//  Copyright © 2020 Pragmatic Programmers. All rights reserved.
//

import Foundation
import SwiftUI

struct TodoList: View {
    // START:ListInfo
    var list: TodoItemList? = nil
    var items: [TodoItem] { list?.allItems ?? defaultTodoItems }
    var title: String { list?.name ?? "All Items" }
    
    func color(for item: TodoItem) -> Color {
        let list = self.list ?? item.list
        return list.color.uiColor
    }
    // END:ListInfo
    
    @State private var sortBy: SortOption = .title
    // START:ShowChooser
    @State private var showingChooser = false
    // END:ShowChooser
    
    var body: some View {
        List(sortedItems) { item in
            // START:NewLinkDestination
            NavigationLink(destination: TodoItemDetail(item: item)) {
                TodoItemRow(item: item)
                    .accentColor(self.color(for: item))
            }
            // END:NewLinkDestination
        }
        .navigationBarTitle("To-Do Items")
        // START:SortBarButton
        .navigationBarItems(trailing: sortButton)
        // END:SortBarButton
        .listStyle(GroupedListStyle())
        // START:ActionSheetCall
        .actionSheet(isPresented: $showingChooser) {
            // <literal:elide> ... </literal:elide>
            // END:ActionSheetCall
            // START:ActionSheetImpl
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.rawValue)) {
                        self.sortBy = opt
                    }
            })
            // END:ActionSheetImpl
            // START:ActionSheetCall
        }
        // END:ActionSheetCall
    }
    
    // START:SortButton
    private var sortButton: some View {
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.square")
                .imageScale(.large)
                .font(.system(size: 24, weight: .bold))
        }
    }
    // END:SortButton
    
    // START:SortOptionEnum
    private enum SortOption: String, CaseIterable {
        case title = "Title"
        case priority = "Priority"
        case dueDate = "Due Date"
    }
    // END:SortOptionEnum
    
    // START:SortItems
    private var sortedItems: [TodoItem] {
        items.sorted {
            switch sortBy {
            case .title:
                return $0.title
                    .caseInsensitiveCompare($1.title) == .orderedAscending
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) < ($1.date ?? .distantFuture)
            }
        }
    }
    // END:SortItems
}

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        NavigationView {
            TodoList()
        }
    }
}
